unit Unrar;
(*
  TUnrar for Kylix ( tested on Kylix3 ) for libunrar.so
  libunrar.so "translated" from unrar.dll by le_barbu.
  libunrar.so Released under rarlab licence (see <www.rarlab.com>).
  TUnrar released under GNU/GPL licence.
  Please report me bugs, wishes and suggestions !
  eMail:  le.barbu.li@free.fr
*)
interface

uses
  SysUtils, Classes, CustomRar, DateUtils, PwdDlg, VolDlg, OverDlg, IniFiles, Libc, Math;

type

  // Testing is the same as listing - so check the file-status in the event OnFileProcessing
  TRarMode = (RARM_EXTRACT, RARM_LIST, RARM_TEST);
  // Used for Status Messages in the event OnStatus
  TRarStatus = (RAR_ONOPEN, RAR_ONBEFOREOPEN,
                RAR_AFTERCLOSE, RAR_ONPASSWORD);
  // Options for overriding files
  TOverrideOptions = (OR_ALL, OR_NONE, OR_ASK);

  RARHeaderDataT  = record
    ArchiveName : string;       // Archive Name
    FileName    : string;       // FileName in Archiv with relativ Path
    PackSize    : cardinal;     // packed filesize
    UnpVer      : Cardinal;     // rar version
    UnpSize     : cardinal;     // unpacked filesize
    HostOS      : string;       // Name of Host Operation System
    FileCRC     : string;       // CRC-Code of File as 'F4F5F6F7'
    FileTime    : TDateTime;    // FileTime (Delphi-Format)
    Method      : String;       // Compress Method
  end;

  TRARProgress = procedure(Sender: TObject; FilesProcessed, FileCount, SizeProcessed, SizeCount: Cardinal) of object;
  TRAROverrideEvent = procedure(Sender: TObject; FileName: string; var CanOverride: boolean) of object;
  TRarStatusEvent = procedure(Sender: TObject; msg: string; status: TRarStatus) of object;
  TRarChangeVolEvent = procedure(Sender: TObject; ArcName: PChar; Mode: integer) of object;
  TRarPromptPassEvent = procedure(Sender: TObject; var Password: string) of object;
  TRarErrorEvent = procedure(Sender: TObject; msg: string; MessageID: integer) of object;
  TRarHeaderEvent = procedure(Sender: TObject; hdrData: RARHeaderDataT; status: Integer) of object;


  TUnrar = class(TComponent)
  private
    { Private declarations }
    hdrData           : RARHeaderData;            // Original HeaderData
    hdrDataT          : RARHeaderDataT;           // Translated HeaderData
    arcData           : RAROpenArchiveData;       // Archive Structure
    RAROpenMode       : Cardinal;                 // Open Mode RAR_EXTRACT or RAR_TEST
    Handle            : THandle;                  // Archive handle
    FirstVolume       : Boolean;                  // if FirstVolume is true then show VolDialog -;- else it is that the first volume event is for the First Volume So we don't need

    FDllVersion       : Integer;
    FPwdForAll        : Boolean;                  // Flag to set same Password for all files of archive
    FDirectory        : String;                   // Target Dir
    FFileName         : String;                   // Archive to extract
    FMode             : TRarMode;                 // Opening mode
    FOverrideEvent    : TOverrideOptions;         // Override mode
    FFileCount        : Cardinal;                 // Number of files counted in archive
    FFilesProcessed   : Cardinal;
    FSizeCount        : Cardinal;                 // Size counted
    FSizeProcessed    : Cardinal;
    FPassword         : String;                   // Password if needed
    FStopProcessing   : Boolean;                  // Do we stop processing ?
    FOnOverride       : TRAROverrideEvent;        // EvenTHandler if FOverrideEvent is OR_EVENT
    FOnPassword       : TRarPromptPassEvent;      // EvenTHandler for password
    FOnProgress       : TRARProgress;             // EvenTHandler for Progress
    FOnError          : TRarErrorEvent;           // EvenTHandler for Errors
    FOnRarStatus      : TRarStatusEvent;          // EvenTHandler for status messages
    FOnVolChange      : TRarChangeVolEvent;       // EvenTHandler if new Volumn needed (if UnRar.dll cant find it automaticly)
    FOnFileProcessing : TRarHeaderEvent;          // EvenTHandler for processing one file
    FCanProgress      : boolean;                  // Use Progress - takes another UnRar-Operation for calculating file count and file size

    function  DoUnRarException(code:integer):RAR_Exception;
    procedure ConvertHeader;
    function  DoUnRarCallBack(msg: Cardinal; UserData, P1, P2: Longint): integer;
    procedure DoStatus(msg: string; status: TRarStatus);
    procedure DoError(msg: string; MessageID: Integer);
    procedure InitRARArchive;
    procedure OpenRARArchive;
    procedure CloseRARArchive;
    procedure SetRARPassword;
    procedure ProcessFileHeader(ReadFileHeaderResult: integer);
    function  ProcessFile(hArcData: THandle; Operation: Integer; DestPath, DestName: PChar): Integer;
    procedure ShowPasswordDialog(var Passwd: string);
    procedure ShowOverrideDialog(Name: String; var CanOverride:Boolean);
    function  ShowPromptDialog(OldVolName: string; NewVolName: PChar): boolean;
    procedure CalcProgress;
    function  Test: boolean;
    function  ExtractFileList(List:TStringList):Boolean;
    procedure SetDirectory(Dir:String);
  protected
    { Protected declarations }
  public
    { Public declarations }
    constructor Create(AOwner:TComponent);override;
    destructor  Destroy;override;
    function    Execute(List:TStringList=nil):Boolean;
  published
    { Published declarations }
    property Directory: string read FDirectory write SetDirectory;
    property FileName: string read FFileName write FFileName;
    property Mode: TRarMode read FMode write FMode default RARM_EXTRACT;
    property OverrideEvent: TOverrideOptions read FOverrideEvent write FOverrideEvent;
    property Password: string read FPassword write FPassword;
    property PasswordForAll:Boolean read FPwdForAll write FPwdForAll default true;
    property StopProcessing: boolean read FStopProcessing write FStopProcessing default false;
    property OnFileProcessing: TRarHeaderEvent read FOnFileProcessing write FOnFileProcessing;
    property CanProgress: boolean read FCanProgress write FCanProgress;
    property OnOverride: TRAROverrideEvent read FOnOverride write FOnOverride;
    property OnPassword: TRarPromptPassEvent read FOnPassword write FOnPassword;
    property OnError: TRarErrorEvent read FOnError write FOnError;
    property OnProgress: TRARProgress read FOnProgress write FOnProgress;
    property OnRarStatus: TRarStatusEvent read FOnRarStatus write FOnRarStatus;
    property OnVolChange: TRarChangeVolEvent read FOnVolChange write FOnVolChange;
    property DllVersion:Integer read FDllVersion;
  end;

procedure Register;

implementation

{$R *.res}

var
  MySelf:Pointer;

procedure Register;
begin
  RegisterComponents('Compression', [TUnrar]);
end;

function UnRarCallBack(msg: Cardinal; UserData, P1, P2: Longint): integer;  cdecl;
begin
  Result  :=  TUnrar(MySelf).DoUnRarCallBack(msg, UserData, P1, P2);
end;

constructor TUnrar.Create(AOwner:TComponent);
begin
  inherited;
  LoadLib;
  if not IsLoaded then
    raise DoUnRarException(ERAR_NO_DLL);
  Mode           :=  RARM_EXTRACT;
  StopProcessing :=  false;
  PasswordForAll:=true;
  FirstVolume:=false;
  FDllVersion:=RARGetDllVersion;
end;

destructor TUnrar.Destroy;
begin
  if IsLoaded then
    FreeLib;
  MySelf  :=  nil;
  inherited;
end;

function TUnrar.DoUnRarException(code:Integer):RAR_Exception;
begin
  Result:=RAR_Exception.Create;
  Result.codeError:=code;
end;

procedure TUnrar.ConvertHeader;
var
  dos2Unix: tm;
begin
  hdrDataT.FileName     :=  String(hdrData.FileName);
  hdrDataT.ArchiveName  :=  String(hdrData.ArcName);
  hdrDataT.PackSize     :=  hdrData.PackSize;
  hdrDataT.UnpVer       :=  hdrData.UnpVer;
  hdrDataT.UnpSize      :=  hdrData.UnpSize;
  hdrDataT.FileCRC      :=  Format('%x', [hdrData.FileCRC]);
  
  case hdrData.HostOS of
    0:  hdrDataT.HostOS :=  'DOS';
    1:  hdrDataT.HostOS :=  'IBM OS/2';
    2:  hdrDataT.HostOS :=  'Windows';
    3:  hdrDataT.HostOS :=  'Unix';
    4:  hdrDataT.HostOS :=  'MacOS';
  end;

  // Translate the DOS time returned by libunrar in Unix time !
  dos2Unix.tm_sec:=  (hdrData.FileTime and $1F)*2;
  dos2Unix.tm_min:=  (hdrData.FileTime shr 5) and $3F;
  dos2Unix.tm_hour:= (hdrData.FileTime shr 11) and $1F;
  dos2Unix.tm_mday:= (hdrData.FileTime shr 16) and $1F;
  dos2Unix.tm_mon:=  ((hdrData.FileTime shr 21)-1) and $0F;
  dos2Unix.tm_year:= (hdrData.FileTime shr 25)+80;
  dos2Unix.tm_isdst:=-1;
  hdrDataT.FileTime     :=  UnixToDateTime(mktime(dos2Unix));

  case hdrData.Method of
    48: hdrDataT.Method :=  COMPRESSMETHODSTORE;
    49: hdrDataT.Method :=  COMPRESSMETHODFASTEST;
    50: hdrDataT.Method :=  COMPRESSMETHODFAST;
    51: hdrDataT.Method :=  COMPRESSMETHODNORMAL;
    52: hdrDataT.Method :=  COMPRESSMETHODGOOD;
    53: hdrDataT.Method :=  COMPRESSMETHODBEST;
  end;
end;

function TUnrar.DoUnRarCallBack(msg: Cardinal; UserData, P1, P2: Longint): integer;
var
  {UnRarRef: TUnRar;}
  Passwd: string;
begin
  Result:=RAR_SUCCESS;
  {UnRarRef := TUnRar(MySelf);}
  case msg of
    UCM_CHANGEVOLUME: begin
                        case P2 of
                          RAR_VOL_ASK:  begin
                                          if Assigned(FOnVolChange) then
                                            FOnVolChange(self, Pointer(P1), RAR_VOL_ASK)
                                          else if ShowPromptDialog(String(hdrData.ArcName), Pointer(P1)) then
                                            Result:=RAR_SUCCESS
                                          else
                                            Result:=RAR_UNSUCCESS;
                                          if String(Pointer(P1)) = '' then
                                             Result:=RAR_UNSUCCESS;
                                        end;

                          RAR_VOL_NOTIFY: begin
                                            if Assigned(FOnVolChange) then
                                              FOnVolChange(self, Pointer(P1), RAR_VOL_NOTIFY);
                                            Result:=RAR_SUCCESS;
                                          end;

                        end;
                      end;
    UCM_NEEDPASSWORD: begin
                        Passwd:=Password;
                        if (Passwd='') or (not PasswordForAll) then
                          if Assigned(FOnPassword) then
                            FOnPassword(self, Passwd)
                          else ShowPasswordDialog(Passwd);
                        Result:=RAR_SUCCESS;
                        StrPCopy(Pointer(P1), Copy(Passwd, 1, P2));
                        DoStatus(Password, RAR_ONPASSWORD);
                      end;

    UCM_PROCESSDATA:  begin
                        // never used - use OnFileProcessing instead
                        // Size := P2;
                        if StopProcessing then
                          Result := RAR_UNSUCCESS
                        else
                          Result := RAR_SUCCESS;
                      end;
  end;
  if StopProcessing then
    Result := RAR_UNSUCCESS;
end;

procedure TUnrar.DoStatus(msg: string; status: TRarStatus);
begin
  if Assigned(FOnRarStatus) then
    FOnRarStatus(self, msg, status);
end;

procedure TUnrar.DoError(msg: string; MessageID: Integer);
begin
  if Assigned(FOnError) then
    FOnError(self, msg, MessageID);
end;

procedure TUnrar.InitRARArchive;
begin
  arcData.ArcName:=PChar(FileName);
  arcData.OpenResult:=RAR_SUCCESS;
  if Mode=RARM_EXTRACT then begin
    RAROpenMode:=RAR_EXTRACT;
    arcData.OpenMode:=RAR_OM_EXTRACT;
  end
  else if Mode=RARM_TEST then begin
    RAROpenMode:=RAR_TEST;
    arcData.OpenMode:=RAR_OM_EXTRACT;
  end
  else if Mode=RARM_LIST then begin
    RAROpenMode:=RAR_EXTRACT;
    arcData.OpenMode:=RAR_OM_LIST;
  end;
  GetMem(arcData.CmtBuf, MAXRARCOMMENTSIZE);
  arcData.CmtBufSize:=MAXRARCOMMENTSIZE;
  arcData.CmtState:=0;
  arcData.CmtSize:=MAXRARCOMMENTSIZE;
end;

procedure TUnrar.OpenRARArchive;
begin
  DoStatus('', RAR_ONBEFOREOPEN);
  Handle := RAROpenArchive(@arcData);
  DoStatus('', RAR_ONOPEN);
  case arcData.CmtState of
    ERAR_NO_MEMORY      : DoError(MSG_EMEMORY, ERAR_NO_MEMORY);
    ERAR_BAD_DATA       : DoError(MSG_ECORRUPT, ERAR_BAD_DATA);
    ERAR_UNKNOWN_FORMAT : DoError(MSG_EUNKNOWN, ERAR_UNKNOWN_FORMAT);
    ERAR_SMALL_BUF      : DoError(MSG_EBUFFER, ERAR_SMALL_BUF);
  end;
end;

procedure TUnrar.CloseRARArchive;
begin
  if RARCloseArchive(Handle) = ERAR_ECLOSE then
    DoError(MSG_ECLOSE, ERAR_ECLOSE);
  DoStatus('', RAR_AFTERCLOSE);
end;

procedure TUnrar.SetRARPassword;
begin
  if Password<>'' then
    RARSetPassword(Handle, PChar(Password));
end;

procedure TUnrar.ProcessFileHeader(ReadFileHeaderResult: integer);
begin
  if ReadFileHeaderResult = ERAR_BAD_DATA then
    DoError(MSG_ECORRUPT, ERAR_BAD_DATA)
  else
  begin
    if assigned(FOnFileProcessing) or assigned(FOnProgress) then
    begin
      ConvertHeader;
      if assigned(FOnFileProcessing) then
        FOnFileProcessing(self, hdrDataT, ReadFileHeaderResult);
      if assigned(FOnProgress) then
      begin
        Inc(FFilesProcessed);
        FSizeProcessed:=FSizeProcessed+hdrData.UnpSize;
        FOnProgress(self, FFilesProcessed, FFileCount, FSizeProcessed, FSizeCount);
      end;
    end;
  end;
end;

function TUnrar.ProcessFile(hArcData: THandle; Operation: Integer; DestPath, DestName: PChar): Integer;
var
  CanOverride:Boolean;
  FileName:String;
begin
  Result:=0;
  FileName:=IncludeTrailingPathDelimiter(String(DestPath))+String(DestName);
  if FOverrideEvent = OR_ALL then
    Result := RARProcessFile(Handle, Operation, PChar(''), PChar(FileName))
  else if FileExists(FileName) and (arcData.OpenMode<>RAR_OM_LIST) then
    case FOverrideEvent of
      OR_NONE : Result:=RARProcessFile(Handle, RAR_SKIP, PChar(''), PChar(FileName));
      OR_ASK  : begin
                  CanOverride:=false;
                  if assigned(FOnOverride) then
                    FOnOverride(self, FileName, CanOverride)
                  else ShowOverrideDialog(FileName, CanOverride);
                  if CanOverride then
                    Result := RARProcessFile(Handle, Operation, PChar(''), PChar(FileName))
                  else
                    Result := RARProcessFile(Handle, RAR_SKIP, PChar(''), PChar(FileName));
                end;
    end
  else
    Result := RARProcessFile(Handle, Operation, PChar(''), PChar(FileName));
end;

procedure TUnrar.ShowPasswordDialog(var Passwd: string);
var
  dlg:TPwdDlg;
begin
  dlg:=TPwdDlg.Create(self);
  dlg.ShowModal;
  Passwd:=PwdPwd;
  PasswordForAll:=PwdSfa;
end;

function TUnrar.ShowPromptDialog(OldVolName: string; NewVolName: PChar): boolean;
var
  dlg:TVolDlg;
begin
  if FirstVolume=false then begin
    FirstVolume:=true;
    Result:=false;
    exit;
  end;
  dlg:=TVolDlg.Create(self, OldVolName);
  dlg.ShowModal;
  if (VolFileName<>'') and (VolFileName<>OldVolName) then begin
    StrPCopy(NewVolName, VolFileName);
    Result:=true;
  end
  else
    Result:=false;
end;

procedure TUnrar.ShowOverrideDialog(Name: String; var CanOverride:Boolean);
var
  dlg:TOverDlg;
begin
  if DirectoryExists(Name) then begin
    CanOverride:=true;
    exit;
  end;
  dlg:=TOverDlg.Create(self);
  dlg.ShowModal;
  CanOverride:=OverResult;
  case OverResult of
    true: if OverAll then
            FOverrideEvent:=OR_ALL
          else
            FOverrideEvent:=OR_ASK;
    false:if OverAll then
            FOverrideEvent:=OR_NONE;
          else
            FOverrideEvent:=OR_ASK;
  end;
end;

procedure TUnrar.CalcProgress;
var
  ReadFileHeaderResult: integer;
  ReadFileResult: Integer;
begin
  FFileCount := 0;
  FSizeCount := 0;
  InitRARArchive;
  arcData.OpenMode:=RAR_OM_EXTRACT;

  if FStopProcessing then
    exit;

  OpenRARArchive;

  try
    if FStopProcessing then
      exit;

    RARSetCallback(Handle, UnRarCallBack, 0);
    SetRARPassword;

    ReadFileResult := RAR_SUCCESS;
    repeat
      ReadFileHeaderResult := RARReadHeader(Handle, @hdrData);
      if ReadFileHeaderResult = ERAR_END_ARCHIVE then
        break;

      if FStopProcessing then
        exit;

      if ReadFileHeaderResult = RAR_SUCCESS then
      begin
        ReadFileResult := RARProcessFile(Handle, RAR_SKIP, PChar(Directory), nil);
        if ReadFileResult = RAR_SUCCESS then
        begin
          if not ((hdrData.Flags and $00000001) = $00000001) then
          begin
            Inc(FFileCount);
            FSizeCount := FSizeCount + hdrData.UnpSize;
          end;
        end;
      end;

      if StopProcessing then
        exit;
    until (ReadFileResult <> RAR_SUCCESS);
  finally
    CloseRARArchive;
  end;
end;

function TUnrar.Test:Boolean;
var
  ReadFileHeaderResult: integer;
  ReadFileResult: Integer;
begin
  Result := false;

  InitRARArchive;
  arcData.OpenMode:=RAR_OM_EXTRACT;

  if FStopProcessing then
    exit;

  OpenRARArchive;
  try
    if FStopProcessing then
      exit;

    RARSetCallback(Handle, UnRarCallBack, 0);
    SetRARPassword;

    ReadFileResult := RAR_SUCCESS;
    repeat
      ReadFileHeaderResult := RARReadHeader(Handle, @hdrData);
      if ReadFileHeaderResult = ERAR_END_ARCHIVE then
        break;

      if FStopProcessing then
        exit;

      if ReadFileHeaderResult = RAR_SUCCESS then
        ReadFileResult := ProcessFile(Handle, RAR_TEST, PChar(Directory), nil);

      if StopProcessing then
        exit;
    until (ReadFileResult <> RAR_SUCCESS);
    Result:=true;
  finally
    CloseRARArchive;
  end;
end;

function TUnrar.Execute(List:TStringList=nil):Boolean;
var
  ReadFileHeaderResult: integer;
  ReadFileResult: Integer;
begin
  MySelf := self;
  Result:=false;

  if List<>nil then begin
    Result:=ExtractFileList(List);
    exit;
  end;

  if Mode=RARM_TEST then begin
    Result:=Test;
    exit;
  end;

  StopProcessing := false;

  FFilesProcessed := 0;
  FSizeProcessed := 0;
  if FCanProgress then
    CalcProgress;

  InitRARArchive;
  if FStopProcessing then
    exit;

  OpenRARArchive;
  try
    if FStopProcessing then
      exit;

    RARSetCallback(Handle, UnRarCallBack, 0);

    SetRARPassword;

    ReadFileResult := RAR_SUCCESS;
    repeat
      // TEST
      hdrData.FileTime:=0;
      ReadFileHeaderResult := RARReadHeader(Handle, @hdrData);
      if ReadFileHeaderResult = ERAR_END_ARCHIVE then
        break;
      ProcessFileHeader(ReadFileHeaderResult);
      if FStopProcessing then
        exit;

      if ReadFileHeaderResult = RAR_SUCCESS then
        ReadFileResult := ProcessFile(Handle, RAROpenMode, PChar(Directory), hdrData.FileName);
      case ReadFileResult of
        ERAR_BAD_DATA       : DoError(MSG_ECORRUPT, ERAR_BAD_DATA);
        ERAR_BAD_ARCHIVE    : DoError(MSG_EARCHIVE, ERAR_BAD_ARCHIVE);
        ERAR_UNKNOWN_FORMAT : DoError(MSG_EUNKNOWN, ERAR_UNKNOWN_FORMAT);
        ERAR_EOPEN          : DoError(MSG_EOPEN, ERAR_EOPEN);
        ERAR_ECREATE        : DoError(MSG_ECREAT, ERAR_ECREATE);
        ERAR_ECLOSE         : DoError(MSG_ECLOSE, ERAR_ECLOSE);
        ERAR_EREAD          : DoError(MSG_EREAD, ERAR_EREAD);
        ERAR_EWRITE         : DoError(MSG_EWRITE, ERAR_EWRITE);
      end;

      if StopProcessing then
        exit;
    // alternativ y can try to unpack the next file and check only for ERAR_END_ARCHIVE
    until (ReadFileResult <> RAR_SUCCESS);
    Result:=true;
  finally
    CloseRARArchive;
  end;
end;

procedure TUnrar.SetDirectory(Dir:String);
begin
  if Dir<>'' then
    FDirectory:=IncludeTrailingPathDelimiter(Dir)
  else
    FDirectory:='';
end;

function TUnrar.ExtractFileList(List:TStringList):Boolean;
var
  ReadFileHeaderResult: integer;
  ReadFileResult: Integer;
begin
  MySelf := self;
  Result:=false;

  InitRARArchive;
  if FStopProcessing then
    exit;

  OpenRARArchive;
  try
    if FStopProcessing then
      exit;

    RARSetCallback(Handle, UnRarCallBack, 0);

    SetRARPassword;

    ReadFileResult := RAR_SUCCESS;
    repeat
      ReadFileHeaderResult := RARReadHeader(Handle, @hdrData);
      if ReadFileHeaderResult = ERAR_END_ARCHIVE then
        break;
      ProcessFileHeader(ReadFileHeaderResult);
      if FStopProcessing then
        exit;

      if ReadFileHeaderResult = RAR_SUCCESS then
        if List.IndexOf(String(hdrData.FileName))<>-1 then
          ReadFileResult := ProcessFile(Handle, RAROpenMode, PChar(Directory), hdrData.FileName)
        else
          ReadFileResult := ProcessFile(Handle, RAR_SKIP, PChar(Directory), hdrData.FileName);
      case ReadFileResult of
        ERAR_BAD_DATA       : DoError(MSG_ECORRUPT, ERAR_BAD_DATA);
        ERAR_BAD_ARCHIVE    : DoError(MSG_EARCHIVE, ERAR_BAD_ARCHIVE);
        ERAR_UNKNOWN_FORMAT : DoError(MSG_EUNKNOWN, ERAR_UNKNOWN_FORMAT);
        ERAR_EOPEN          : DoError(MSG_EOPEN, ERAR_EOPEN);
        ERAR_ECREATE        : DoError(MSG_ECREAT, ERAR_ECREATE);
        ERAR_ECLOSE         : DoError(MSG_ECLOSE, ERAR_ECLOSE);
        ERAR_EREAD          : DoError(MSG_EREAD, ERAR_EREAD);
        ERAR_EWRITE         : DoError(MSG_EWRITE, ERAR_EWRITE);
      end;

      if StopProcessing then
        exit;
    // alternativ y can try to unpack the next file and check only for ERAR_END_ARCHIVE
    until (ReadFileResult <> RAR_SUCCESS);
    Result:=true;
  finally
    RARCloseArchive(Handle);
  end;
end;

end.
